/*
 * Copyright (c) 2007, Anthony Minessale II
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * * Neither the name of the original author; nor the names of any contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 * 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The Initial Developer of the Original Code is
 * Anthony Minessale II <anthmct@yahoo.com>
 * Portions created by the Initial Developer are Copyright (C)
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 * 
 * Anthony Minessale II <anthmct@yahoo.com>
 * PeteDao <petekay@gmail.com>
 * Steve Underwood 0.0.1 <steveu@coppice.org>
 * 
 *
 * mod_say_zh.c -- Say for Mandarin, Cantonese, and probably any other Chinese
 *                 dialect.
 *
 */

#include <switch.h>
#include <math.h>
#include <ctype.h>

/*  SSM_COUNTED means ordinal number
	SSM_PRONOUNCED
    SSM_ITERATED */

SWITCH_MODULE_LOAD_FUNCTION(mod_say_zh_load);
SWITCH_MODULE_DEFINITION(mod_say_zh, mod_say_zh_load, NULL, NULL);

#define say_num(num, t) {							\
		char tmp[80];\
		switch_status_t tstatus;\
		switch_snprintf(tmp, sizeof(tmp), "%u", (unsigned)num);				\
	if ((tstatus = zh_say_general_count(session, tmp, SST_ITEMS, t, args)) != SWITCH_STATUS_SUCCESS) {\
		return tstatus;\
	}\
}\

#define say_file(...) {\
		char tmp[80];\
		switch_status_t tstatus;\
		switch_snprintf(tmp, sizeof(tmp), __VA_ARGS__);\
		if ((tstatus = switch_ivr_play_file(session, NULL, tmp, args)) != SWITCH_STATUS_SUCCESS){ \
			return tstatus;\
		}\
		if (!switch_channel_ready(switch_core_session_get_channel(session))) {\
			return SWITCH_STATUS_FALSE;\
		}\
}\


static switch_status_t zh_spell(switch_core_session_t *session, char *tosay, switch_say_type_t type, switch_say_method_t method, switch_input_args_t *args)
{
	char *p;

	for (p = tosay; p && *p; p++) {
		int a = tolower((int) *p);
		if (a >= '0' && a <= '9') {
			say_file("digits/%d.wav", a - '0');
		} else {
			if (type == SST_NAME_SPELLED) {
				say_file("ascii/%d.wav", a);
			} else if (type == SST_NAME_PHONETIC) {
				say_file("phonetic-ascii/%d.wav", a);
			}
		}
	}

	return SWITCH_STATUS_SUCCESS;
}

static char *strip_commas(char *in, char *out, switch_size_t len)
{
	char *p = in;
    char *q = out;
	char *ret = out;
	switch_size_t x = 0;

	for ( ; p && *p; p++) {
		if ((*p >= '0' && *p <= '9')) {
			*q++ = *p;
		} else if (*p != ',') {
			ret = NULL;
			break;
		}

		if (++x > len) {
			ret = NULL;
			break;
		}
	}

	return ret;
}

static char *strip_nonnumerics(char *in, char *out, switch_size_t len)
{
	char *p = in;
    char *q = out;
	char *ret = out;
	switch_size_t x = 0;

	/* valid are 0 - 9, period (.), minus (-), and plus (+) - remove all others */
	for (; p && *p; p++) {
		if ((*p >= '0' && *p <= '9') || *p == '.' || *p == '-' || *p == '+') {
			*q++ = *p;
		}

		if (++x > len) {
			ret = NULL;
			break;
		}
	}

	return ret;
}

static switch_status_t zh_say_general_count(switch_core_session_t *session,
											char *tosay, switch_say_type_t type, switch_say_method_t method, switch_input_args_t *args)
{
	int in;
	char sbuf[13] = "";
    char digits[11];
    int i;

	if (!(tosay = strip_commas(tosay, sbuf, sizeof(sbuf))) || strlen(tosay) > 9) {
		switch_log_printf(SWITCH_CHANNEL_LOG, SWITCH_LOG_ERROR, "Parse Error!\n");
		return SWITCH_STATUS_GENERR;
	}

	in = atoi(tosay);

	if (in != 0) {
		snprintf(digits, sizeof(digits), "%10.10d", in);
		switch (method) {
		case SSM_COUNTED:
			say_file("digits/ordinal.wav");
			/* Fall through */
		case SSM_PRONOUNCED:
            for (i = 0;  i <= 9;  i++)
            {
                switch (i)
                {
                case 0:
                    /* Billions column */
                    if (digits[i] != '0')
                    {
                        if (digits[i] != '1')
                    		say_file("digits/%c.wav", digits[i]);
                  		say_file("digits/10.wav");
                    }
                    break;
                case 1:
                    /* Hundred millions columns */
                    if (digits[i] != '0')
                   		say_file("digits/%c.wav", digits[i]);
                    if (memcmp (digits, "00", 2) != 0)
                  		say_file("digits/100000000.wav");
                    break;
                case 2:
                case 6:
                    /* Ten millions or thousands column */
                    if (digits[i] != '0')
                    {
                   		say_file("digits/%c.wav", digits[i]);
                  		say_file("digits/1000.wav");
                    }
                    break;
                case 3:
                case 7:
                    /* Millions or hundreds column */
                    if (digits[i] != '0')
                    {
                   		say_file("digits/%c.wav", digits[i]);
                   		say_file("digits/100.wav");
                    }
                    break;
                case 4:
                    /* Hundred thousands column */
                    if (digits[i] != '0')
                    {
                        if (digits[i] != '1'
                            ||
                            memcmp (digits, "0000", 4) != 0)
                        {
                       		say_file("digits/%c.wav", digits[i]);
                        }
                    }
                    else
                    {
                        if (digits[i + 1] != '0'
                            &&
                            memcmp (digits, "0000", 4) != 0)
                        {
                       		say_file("digits/%c.wav", digits[i]);
                        }
                    }
                    if (digits[i] != '0')
                   		say_file("digits/10.wav");
                    break;
        
                case 5:
                    /* Ten thousands column */
                    if (digits[i] != '0')
                    {
                   		say_file("digits/%c.wav", digits[i]);
                        if (memcmp (digits + 2, "0000", 4) != 0)
                       		say_file("digits/10000.wav");
                    }
                    break;
                case 8:
                    /* Tens column */
                    if (digits[i] != '0')
                    {
                        if (digits[i] != '1'
                            ||
                            memcmp (digits, "00000000", 8) != 0)
                        {
                       		say_file("digits/%c.wav", digits[i]);
                        }
                    }
                    else
                    {
                        if (digits[9] != '0'
                            &&
                            memcmp (digits, "00000000", 8) != 0)
                        {
                       		say_file("digits/%c.wav", digits[i]);
                        }
                    }
                    if (digits[8] != '0')
                   		say_file("digits/10.wav");
                    break;
                case 9:
                    /* Units column */
                    if (digits[9] != '0')
                   		say_file("digits/%c.wav", digits[i]);
                    break;
                }
            }
			break;
		case SSM_ITERATED:
			{
				char *p;

				for (p = tosay; p && *p; p++) {
					say_file("digits/%c.wav", *p);
				}
			}
			break;
		default:
			break;
		}
	} else {
		say_file("digits/0.wav");
	}

	return SWITCH_STATUS_SUCCESS;
}


static switch_status_t zh_ip(switch_core_session_t *session, char *tosay, switch_say_type_t type, switch_say_method_t method, switch_input_args_t *args)
{
	char *a;
    char *b;
    char *c;
    char *d;

	switch_status_t status = SWITCH_STATUS_SUCCESS;
	if (!(a = strdup(tosay))) {
		return SWITCH_STATUS_FALSE;
	}

	if (!(b = strchr(a, '.'))) {
		status = SWITCH_STATUS_FALSE;
		goto done;
	}

	*b++ = '\0';

	if (!(c = strchr(b, '.'))) {
		status = SWITCH_STATUS_FALSE;
		goto done;
	}

	*c++ = '\0';

	if (!(d = strchr(c, '.'))) {
		status = SWITCH_STATUS_FALSE;
		goto done;
	}

	*d++ = '\0';

	say_num(atoi(a), method);
	say_file("digits/dot.wav");
	say_num(atoi(b), method);
	say_file("digits/dot.wav");
	say_num(atoi(c), method);
	say_file("digits/dot.wav");
	say_num(atoi(d), method);

done:
	switch_safe_free(a);
	return status;
}

static switch_status_t zh_say_time(switch_core_session_t *session,
								   char *tosay,
								   switch_say_type_t type,
								   switch_say_method_t method,
								   switch_input_args_t *args)
{
	int32_t t;
	switch_time_t target = 0;
	switch_time_exp_t tm;
	uint8_t say_date = 0;
    uint8_t say_time = 0;

	if (type == SST_TIME_MEASUREMENT) {
		int64_t hours = 0;
		int64_t minutes = 0;
		int64_t seconds = 0;
		int64_t r = 0;

		if (strchr(tosay, ':')) {
			char *tme = switch_core_session_strdup(session, tosay);
			char *p;

			if ((p = strrchr(tme, ':'))) {
				*p++ = '\0';
				seconds = atoi(p);
				if ((p = strchr(tme, ':'))) {
					*p++ = '\0';
					minutes = atoi(p);
					if (tme) {
						hours = atoi(tme);
					}
				} else {
					minutes = atoi(tme);
				}
			}
		} else {
			if ((seconds = atoi(tosay)) <= 0) {
				seconds = (int64_t) switch_timestamp(NULL);
			}

			if (seconds >= 60) {
				minutes = seconds / 60;
				r = seconds % 60;
				seconds = r;
			}

			if (minutes >= 60) {
				hours = minutes / 60;
				r = minutes % 60;
				minutes = r;
			}
		}

		if (hours) {
			if (hours == 2) {
				say_file("time/2s.wav");
 			} else {
			    say_num(hours, SSM_PRONOUNCED);
			}
			say_file("time/hours.wav");
		}
		if (minutes) {
			if (minutes == 2) {
				say_file("time/2s.wav");
 			} else {
			    say_num(minutes, SSM_PRONOUNCED);
			}
			say_file("time/minute.wav");
		} else {
			if (hours) {
				say_file("digits/0.wav");
				say_file("time/minute.wav");
			}
		}

		if (seconds) {
			if (seconds == 2) {
				say_file("time/2s.wav");
 			} else {
			    say_num(hours, SSM_PRONOUNCED);
			}
			say_file("time/seconds.wav");
		} else {
			if (hours  ||  minutes) {
				say_file("digits/0.wav");
				say_file("time/seconds.wav");
			}
		}

		return SWITCH_STATUS_SUCCESS;
	}

	if ((t = atoi(tosay)) > 0)
		target = switch_time_make(t, 0);
	else
		target = switch_timestamp_now();
	switch_time_exp_lt(&tm, target);

	switch (type) {
	case SST_CURRENT_DATE_TIME:
		say_date =
		say_time = 1;
		break;
	case SST_CURRENT_DATE:
		say_date = 1;
		break;
	case SST_CURRENT_TIME:
		say_time = 1;
		break;
	default:
		break;
	}

	if (say_date) {
        say_num(tm.tm_year + 1900, SSM_ITERATED);
		say_file("time/year.wav");
		say_num(tm.tm_mon + 1, SSM_PRONOUNCED);
		say_file("time/month.wav");
		say_num(tm.tm_mday, SSM_PRONOUNCED);
		say_file("time/day.wav");
		say_file("time/day-%d.wav", tm.tm_wday);
	}

	if (say_time) {
		int32_t hour = tm.tm_hour;

		if (hour < 6) {
            say_file("time/morning.wav");
		} else if (hour < 12) {
            say_file("time/a-m.wav");
		} else if (tm.tm_hour < 18) {
            say_file("time/p-m.wav");
		} else {
            say_file("time/evening.wav");
		}
		if (hour > 12)
			hour -= 12;
		if (hour == 0)
			hour = 12;

		if (hour == 2) {
			say_file("time/2s.wav");
		} else {
			say_num(hour, SSM_PRONOUNCED);
		}
		say_file("time/hour.wav");
		if (tm.tm_min == 2) {
			say_file("time/2s.wav");
		} else {
			say_num(tm.tm_min, SSM_PRONOUNCED);
		}
		say_file("time/minute.wav");  
	}
	return SWITCH_STATUS_SUCCESS;
}

static switch_status_t zh_say_money(switch_core_session_t *session, char *tosay, switch_say_type_t type, switch_say_method_t method,
									switch_input_args_t *args)
{
	char sbuf[16] = "";			/* enough for 999,999,999,999.99 (w/o the commas or leading $) */
	char *dollars = NULL;
	char *cents = NULL;

	if (strlen(tosay) > 15 || !(tosay = strip_nonnumerics(tosay, sbuf, sizeof(sbuf)))) {
		switch_log_printf(SWITCH_CHANNEL_LOG, SWITCH_LOG_ERROR, "Parse Error!\n");
		return SWITCH_STATUS_GENERR;
	}

	dollars = sbuf;

	if ((cents = strchr(sbuf, '.'))) {
		*cents++ = '\0';
		if (strlen(cents) > 2) {
			cents[2] = '\0';
		}
	}

	/* If positive sign - skip over" */
	if (sbuf[0] == '+') {
		dollars++;
	}

	/* If negative say "negative" */
	if (sbuf[0] == '-') {
		say_file("currency/negative.wav");
		dollars++;
	}

	/* Say dollar amount */
	zh_say_general_count(session, dollars, type, method, args);
	say_file("currency/dollar.wav");

	/* Say cents */
	if (cents) {
		zh_say_general_count(session, cents, type, method, args);
	} else {
		say_file("digits/0.wav");
	}
	say_file("currency/cent.wav");

	return SWITCH_STATUS_SUCCESS;
}

static switch_status_t zh_say(switch_core_session_t *session, char *tosay, switch_say_type_t type, switch_say_method_t method, switch_input_args_t *args)
{
	switch_say_callback_t say_cb = NULL;

	switch (type) {
	case SST_NUMBER:
	case SST_ITEMS:
	case SST_PERSONS:
	case SST_MESSAGES:
		say_cb = zh_say_general_count;
		break;
	case SST_TIME_MEASUREMENT:
	case SST_CURRENT_DATE:
	case SST_CURRENT_TIME:
	case SST_CURRENT_DATE_TIME:
		say_cb = zh_say_time;
		break;
	case SST_IP_ADDRESS:
		say_cb = zh_ip;
		break;
	case SST_NAME_SPELLED:
	case SST_NAME_PHONETIC:
		say_cb = zh_spell;
		break;
	case SST_CURRENCY:
		say_cb = zh_say_money;
		break;
	default:
		switch_log_printf(SWITCH_CHANNEL_LOG, SWITCH_LOG_ERROR, "Unknown Say type=[%d]\n", type);
		break;
	}

	if (say_cb) {
		return say_cb(session, tosay, type, method, args);
	}

	return SWITCH_STATUS_FALSE;
}

SWITCH_MODULE_LOAD_FUNCTION(mod_say_zh_load)
{
	switch_say_interface_t *say_interface;
	/* connect my internal structure to the blank pointer passed to me */
	*module_interface = switch_loadable_module_create_module_interface(pool, modname);
	say_interface = switch_loadable_module_create_interface(*module_interface, SWITCH_SAY_INTERFACE);
	say_interface->interface_name = "zh";
	say_interface->say_function = zh_say;

	/* indicate that the module should continue to be loaded */
	return SWITCH_STATUS_SUCCESS;
}

/* For Emacs:
 * Local Variables:
 * mode:c
 * indent-tabs-mode:t
 * tab-width:4
 * c-basic-offset:4
 * End:
 * For VIM:
 * vim:set softtabstop=4 shiftwidth=4 tabstop=4 expandtab:
 */
